/*
 * Crestron Electronics
 * 
 * KeyButton uiHal interface
 *
 * Raymond Kuschan
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/input/matrix_keypad.h>
#include <linux/delay.h>
#include <linux/signal.h>
#include <asm/siginfo.h>        //siginfo

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#include <linux/crestron.h>

#define PROC_CPUFREQ_FILENAME           "cpuFrequency"
#define PROCFS_MAX_SIZE     1024


/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *CpuFreq_File;

/** * The buffer (2k) for this module */
char gCpuFreq_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
unsigned long gCpuFreq_buffer_size = 0;


struct cpufreq *st_cpufreq;

//extern struct mutex *cpuFreqMutex; //mutex lock for gCpufreq stat
	
extern int uiHal_setCpuFreq(int buf);

/** * This funtion is called when the /proc hardkey file is read */
static ssize_t CpuFreq_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  static int finished = 0;
  char * p = gCpuFreq_buffer;
  int value;


  /* needed to stop from continuously printing */
  if ( finished == 1 ) { finished=0; return 0; }
  finished = 1;

  mutex_lock(&st_cpufreq->cpuFreqMutex);

   p += sprintf ( p, "FREQUENCY=%d\n" , st_cpufreq->CpuFreq);
   p += sprintf ( p, "VOLTAGE=%d\n" , st_cpufreq->CpuVolt);
   p += sprintf ( p, "SETPOINT=%d\n" , st_cpufreq->CpuSetPt);

  mutex_unlock(&st_cpufreq->cpuFreqMutex);

   gCpuFreq_buffer_size = p-gCpuFreq_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gCpuFreq_buffer, gCpuFreq_buffer_size) ) {
                return -EFAULT;
        }

  return gCpuFreq_buffer_size;
}

static ssize_t CpuFreq_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;
        u8 regVal=0;

        gCpuFreq_buffer_size = len;
        if (gCpuFreq_buffer_size > PROCFS_MAX_SIZE ) {
                gCpuFreq_buffer_size = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(gCpuFreq_buffer, buffer, gCpuFreq_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

	if ( strcmp ( tag, "FREQUENCY" ) == 0 )
        {
                value = strsep ( tempPtr, "=" );
                sscanf ( value, "%d", &st_cpufreq->new_CpuFreq );
                switch(st_cpufreq->new_CpuFreq) {
                        case 1000000000:
                                uiHal_setCpuFreq(1); 
                                break;
		   // commented rates are currently unsupported
                   //     case 900000000:
                   //             uiHal_setCpuFreq(2); 
                   //             break;
                        case 800000000:
                                uiHal_setCpuFreq(3);
                                break;
                   //     case 600000000:
                   //             uiHal_setCpuFreq(4);
                   //             break;
                        case 400000000:
                                uiHal_setCpuFreq(5);
                                break;
                        case 333000000:
                                uiHal_setCpuFreq(6);
                                break;
                   //     case 167000000:
                   //             uiHal_setCpuFreq(7);
                   //             break;
                        default:
                                printk(KERN_ERR "unKnown cpu frequency value. (%d)\n", st_cpufreq->new_CpuFreq);
                                return -1;
                        }
        }

        if ( strcmp ( tag, "SETPOINT" ) == 0 )
        {
                value = strsep ( tempPtr, "=" );
                sscanf ( value, "%d", &st_cpufreq->new_CpuFreq );

			if (st_cpufreq->new_CpuFreq > 0 ||
				st_cpufreq->new_CpuFreq < 8 )
			{
				uiHal_setCpuFreq(st_cpufreq->new_CpuFreq);
			}
			else
			{
				printk(KERN_ERR "unKnown cpu setpoint value. (%d)\n", st_cpufreq->new_CpuFreq);	
				return -1;
			}
        }

        return gCpuFreq_buffer_size;
}
/*
 * This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}
/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int CpuFreq_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int CpuFreq_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_CpuFreq_File = {
        .read    = CpuFreq_read,
        .write   = CpuFreq_write,
        .open    = CpuFreq_open_procfs,
        .release = CpuFreq_close_procfs,
};
/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
        .permission = module_permission,        /* check for permissions */
};

int uiCpuFreqIntfcInit(void)
{
        int error;

/* create the /proc file for CpuFreq bitmask */
        CpuFreq_File = create_proc_entry(PROC_CPUFREQ_FILENAME, 0644, NULL);
        if (CpuFreq_File == NULL){
                printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
                       PROC_CPUFREQ_FILENAME);
                return -ENOMEM;
        }
        else
        {
          //CpuFreq_File->owner = THIS_MODULE;
          CpuFreq_File->proc_iops = &Inode_Ops_File;
          CpuFreq_File->proc_fops = &File_Ops_CpuFreq_File;
          CpuFreq_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
          CpuFreq_File->uid = 0;
          CpuFreq_File->gid = 0;
          CpuFreq_File->size = 80;
       }

        return error;
}

